/***************************************************************************
 *
 * Copyright (c) 2014 Codethink Limited
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ****************************************************************************/

#include <signal.h>
#include <sys/signalfd.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <poll.h>
#include <errno.h>

#include <string>
#include <iostream>
#include <sstream>
#include <fstream>

#include "ilm_types.h"

#include "egl_helper.h"
#include "gles2application.h"

using namespace std;

enum dump_pixel_format
{
    DUMP_INVALID = 0,
    DUMP_GL_RGBA_8888 = 1,
    DUMP_WL_ARGB_8888 = 2,
    DUMP_RGB_565 = 3
};

struct cmdline_args {
    t_ilm_layer layerID;
    t_ilm_surface surfaceID;
    unsigned int width;
    unsigned int height;
    enum dump_pixel_format format;
    unsigned int lpad;
    unsigned int rpad;
    unsigned int tpad;
    unsigned int bpad;
    unsigned int lstrip;
    unsigned int rstrip;
    unsigned int tstrip;
    unsigned int bstrip;
    bool vflip;
    string path;
};

static bool parse_cmdline_args(struct cmdline_args *args, int argc, char **argv)
{
    // Set sensible defaults
    args->layerID = 2400;
    args->surfaceID = 333;
    args->width = 1024;
    args->height = 768;
    args->format = DUMP_GL_RGBA_8888;
    args->lpad = 0;
    args->rpad = 0;
    args->tpad = 0;
    args->bpad = 0;
    args->lstrip = 0;
    args->rstrip = 0;
    args->tstrip = 0;
    args->bstrip = 0;
    args->vflip = false;
    args->path = "EGLWLDump.data";

    for (int i = 1; i < argc; i++)
    {
        string arg_str(argv[i]);
        size_t mid_point = arg_str.find('=');
        string key(arg_str, 0, mid_point);
        string value(arg_str, mid_point + 1);
        cout << "key=" << key << ", value=" << value << endl;
        istringstream iss(value);
        if (key.compare("-layer") == 0)
            iss >> args->layerID;
        else if (key.compare("-surface") == 0)
            iss >> args->surfaceID;
        else if (key.compare("-width") == 0)
            iss >> args->width;
        else if (key.compare("-height") == 0)
            iss >> args->height;
        else if (key.compare("-lpad") == 0)
            iss >> args->lpad;
        else if (key.compare("-rpad") == 0)
            iss >> args->rpad;
        else if (key.compare("-tpad") == 0)
            iss >> args->tpad;
        else if (key.compare("-bpad") == 0)
            iss >> args->bpad;
        else if (key.compare("-lstrip") == 0)
            iss >> args->lstrip;
        else if (key.compare("-rstrip") == 0)
            iss >> args->rstrip;
        else if (key.compare("-tstrip") == 0)
            iss >> args->tstrip;
        else if (key.compare("-bstrip") == 0)
            iss >> args->bstrip;
        else if (key.compare("-path") == 0)
            args->path = value;
        else if (key.compare("-vflip") == 0)
        {
            if (value.compare("true") == 0)
                args->vflip = true;
            else if (value.compare("false") == 0)
                args->vflip = false;
            else
            {
                cerr << "Unexpected vflip '" << value
                     << "', expected values are 'true' and 'false'" << endl;
                return false;
            }
        }
        else if (key.compare("-format") == 0)
        {
            if (value.compare("GL_RGBA_8888") == 0)
                args->format = DUMP_GL_RGBA_8888;
            else if (value.compare("WL_ARGB_8888") == 0)
                args->format = DUMP_WL_ARGB_8888;
            else if (value.compare("RGB_565") == 0)
                args->format = DUMP_RGB_565;
            else
            {
                cerr << "Unexpected format '" << value << "', "
                        "expected values are GL_RGBA_8888, WL_ARGB_8888 and "
                        "RGB_565" << endl;
                return false;
            }
        }
    }
    return true;
}

int formatToDepth(enum dump_pixel_format format)
{
    switch(format)
    {
        case DUMP_GL_RGBA_8888:
        case DUMP_WL_ARGB_8888:
            return 4;
        case DUMP_RGB_565:
            return 2;
        default:
            cerr << "Unrecognized pixel format " << format << endl;
            return 0;
    }
}

bool writeOutputBuffer(char *input_buffer, char *output_buffer,
                       enum dump_pixel_format format, unsigned int size)
{

    if (format == DUMP_GL_RGBA_8888)
    {
        // Output buffer is exactly the same format as input buffer
        for (unsigned int i = 0; i < size * 4; i++)
            output_buffer[i] = input_buffer[i];
    }
    else if (format == DUMP_WL_ARGB_8888)
    {
        for (unsigned int i = 0; i < size; i++)
        {
            // swap R and B around
            output_buffer[i * 4 + 2] = input_buffer[i * 4 + 0]; // R
            output_buffer[i * 4 + 1] = input_buffer[i * 4 + 1]; // G
            output_buffer[i * 4 + 0] = input_buffer[i * 4 + 2]; // B
            output_buffer[i * 4 + 3] = input_buffer[i * 4 + 3]; // A

        }
    }
    else if (format == DUMP_RGB_565)
    {
        for (unsigned int i = 0; i < size; i++)
        {
            // Input Stored as RGBA
            uint8_t red = input_buffer[i * 4 + 0];
            uint8_t green = input_buffer[i * 4 + 1];
            uint8_t blue = input_buffer[i * 4 + 2];

            // Output stored as RGB565
            uint16_t *output_ptr = (uint16_t *)(output_buffer + i * 2);
            *output_ptr = 0x0000;
            *output_ptr |= (0x001F & (blue >> 3));
            *output_ptr |= (0x07E0 & ((green >> 2) << 5));
            *output_ptr |= (0xF800 & ((red >> 3) << 11));
        }
    }
    else
    {
        cerr << "Unexpected format " << format << endl;
        return false;
    }
    return true;
}

bool dumpBuffer(struct cmdline_args& args)
{
    EGLint error;
    int depth;
    bool retval = true;
    static const char zero = 0;

    depth = formatToDepth(args.format);
    if (depth == 0)
        return false;


    char *read_buffer = (char *)malloc(args.width * args.height * 4);
    if (read_buffer == NULL)
    {
        cerr << "Failed to allocate memory for read buffer" << endl;
        return false;
    }

    char *buffer = (char *)malloc(args.width * args.height * depth);
    if (buffer == NULL)
    {
        cerr << "Failed to allocate memory for write buffer" << endl;
        free(read_buffer);
        return false;
    }


    glReadPixels(0, 0, args.width, args.height, GL_RGBA, GL_UNSIGNED_BYTE,
                 read_buffer); // TODO: Ponder whether this works with Gal2d
    error = glGetError();
    if (error != GL_NO_ERROR)
    {
        cerr << "Failed to read pixels, code 0x" << hex << error << endl;
        retval = false;
    }
    else
    {

        if (!writeOutputBuffer(read_buffer, buffer, args.format,
                               args.width * args.height))
        {
            cerr << "Failed to write output buffer" << endl;
        }
        else
        {
            unsigned int total_width = args.width + args.lpad + args.rpad
                               - args.lstrip - args.rstrip;
            off_t top_skip = args.tstrip * args.width * depth;
            off_t left_skip = args.lstrip * depth;
            off_t right_skip = args.rstrip * depth;
            size_t width_len = args.width * depth - left_skip - right_skip;
            int stride;
            char *start_pos;

            if (args.vflip == true)
            {
                top_skip = args.bstrip * args.width * depth;
                stride = -args.width * depth;
                start_pos = (char *)buffer + args.width
                                             * (args.height - args.tstrip - 1)
                                             * depth;
            }
            else
            {
                top_skip = args.tstrip * args.width * depth;
                stride = args.width * depth;
                start_pos = (char *)buffer + top_skip + left_skip;
            }

            char *pos = start_pos;
            ofstream outfile(args.path.c_str(), ofstream::binary);
            if (!outfile.good())
            {
                cerr << "Failed to open file " << args.path << endl;
                retval = false;
            }
            else
            {
                // write top padding
                for (unsigned int i = 0; i < total_width * args.tpad * depth; i++)
                    outfile.write(&zero, 1);

                // write each line for the buffer
                for (unsigned int i = 0;
                     i < args.height - args.bstrip - args.tstrip;
                     i++, pos += stride)
                {
                    // write left padding
                    for (unsigned int j = 0; j < args.lpad * depth; j++)
                        outfile.write(&zero, 1);

                    // write line (minus rstrip and lstrip)
                    outfile.write(pos, width_len);

                    // write right padding
                    for (unsigned int j = 0; j < args.rpad * depth; j++)
                        outfile.write(&zero, 1);
                }

                // write bottom padding
                for (unsigned int i = 0; i < total_width * args.bpad * depth; i++)
                    outfile.write(&zero, 1);

                if (outfile.fail())
                {
                    cerr << "Error occurred while writing to file "
                         << args.path << endl;
                    retval = false;
                }
                outfile.close();
            }
        }
    }
    cout << "Finished writing dump" << endl;
    free(buffer);
    free(read_buffer);
    return retval;
}

int main(int argc, char **argv)
{
    struct cmdline_args args;
    int exit_status = EXIT_SUCCESS;
    ilmErrorTypes err;

    // Set signal handler
    sigset_t mask;
    struct signalfd_siginfo fdsi;
    struct pollfd pfd;
    ssize_t bytes_read;
    int ret;

    WLContextStruct *wl_context;
    EglContextStruct *egl_context;

    sigemptyset(&mask);
    sigaddset(&mask, SIGINT);
    sigaddset(&mask, SIGTERM);
    // Block signals from being handled by the default signal handler
    if (sigprocmask(SIG_BLOCK, &mask, NULL) == -1)
    {
        cerr << "Failed to block signals: " << strerror(errno) << endl;
        exit_status = EXIT_FAILURE;
        goto exit;
    }
    pfd.fd = signalfd(-1, &mask, 0);
    if (pfd.fd == -1)
    {
        cerr << "Failed to set signal fd: " << strerror(errno) << endl;
        exit_status = EXIT_FAILURE;
        goto exit;
    }
    pfd.events = POLLIN | POLLERR | POLLHUP;

    // parse args
    if (!parse_cmdline_args(&args, argc, argv))
    {
        exit_status = EXIT_FAILURE;
        goto exit;
    }


    err = ilm_init();
    if (err != ILM_SUCCESS)
    {
        cerr << "Failed to initialize ilm client: "
             << ILM_ERROR_STRING(err) << endl;
        exit_status = EXIT_FAILURE;
        goto exit;
    }

    wl_context = createWLContext(args.width, args.height);

    if (wl_context == NULL)
    {
        cerr << "Failed to create wayland context" << endl;
        exit_status = EXIT_FAILURE;
        goto exit_ilm;
    }

    egl_context = createEGLContext(args.width, args.height, args.layerID,
                                   args.surfaceID, wl_context);
    if (egl_context == NULL)
    {
        cerr << "Failed to create EGL context" << endl;
        exit_status = EXIT_FAILURE;
        goto exit_wlcontext;
    }

    if (!initGlApplication())
    {
        cerr << "Failed to initialize GL application" << endl;
        exit_status = EXIT_FAILURE;
        goto exit_eglcontext;
    }

    draw(0);
    if (!dumpBuffer(args))
    {
        cerr << "Error occurred when dumping buffer to " << args.path << endl;
        exit_status = EXIT_FAILURE;
        goto exit_eglcontext;
    }
    swapBuffers(wl_context, egl_context);

    while (true)
    {
        // Poll the signalfd, and break out of loop if SIGINT or SIGHUP
        // are found.
        ret = poll(&pfd, 1, -1);
        if (ret == 0)
        {
            cerr << "Poll timed out. This should not happen." << endl;
            exit_status = EXIT_FAILURE;
            goto exit_eglcontext;
        }
        else if (ret < 0)
        {
            cerr << "Error while polling: " << strerror(errno) << endl;
            exit_status = EXIT_FAILURE;
            goto exit_eglcontext;
        }
        if (!(pfd.revents & POLLIN))
        {
            cerr << "Poll returned because of error or hangup" << endl;
            exit_status = EXIT_FAILURE;
            goto exit_eglcontext;
        }

        bytes_read = read(pfd.fd, &fdsi, sizeof(fdsi));
        if (bytes_read != sizeof(fdsi))
        {
            cerr << "Short read when getting signal info" << endl;
            exit_status = EXIT_FAILURE;
            goto exit_eglcontext;
        }

        if (fdsi.ssi_signo == SIGINT || fdsi.ssi_signo == SIGTERM)
            break;
    }
    cout << "Terminating example" << endl;

    exit_eglcontext:
        destroyEglContext(egl_context);
    exit_wlcontext:
        destroyWLContext(wl_context);
    exit_ilm:
        ilm_destroy();
    exit:
        exit(exit_status);
}
